/**
 * Expired Domains Checker - Popup Script
 */

const DEFAULT_CONFIG = {
    serverUrl: 'http://localhost:3847',
    geo: 'it',
    autoLoad: false,
    batchSize: 10
};

// Carica impostazioni salvate
async function loadSettings() {
    return new Promise((resolve) => {
        chrome.storage.sync.get(DEFAULT_CONFIG, (config) => {
            document.getElementById('serverUrl').value = config.serverUrl;
            document.getElementById('geo').value = config.geo;
            document.getElementById('batchSize').value = config.batchSize;
            document.getElementById('autoLoad').checked = config.autoLoad;
            resolve(config);
        });
    });
}

// Salva impostazioni
async function saveSettings() {
    const config = {
        serverUrl: document.getElementById('serverUrl').value.trim() || DEFAULT_CONFIG.serverUrl,
        geo: document.getElementById('geo').value,
        batchSize: parseInt(document.getElementById('batchSize').value),
        autoLoad: document.getElementById('autoLoad').checked
    };

    return new Promise((resolve) => {
        chrome.storage.sync.set(config, () => {
            resolve(config);
        });
    });
}

// Mostra status
function showStatus(message, isError = false) {
    const status = document.getElementById('status');
    status.textContent = message;
    status.className = 'status ' + (isError ? 'error' : 'success');

    setTimeout(() => {
        status.className = 'status';
    }, 3000);
}

// Testa connessione al server
async function testConnection(serverUrl) {
    try {
        const response = await fetch(`${serverUrl}/health`, {
            method: 'GET',
            timeout: 5000
        });

        if (!response.ok) throw new Error(`HTTP ${response.status}`);

        const data = await response.json();
        return data.status === 'ok';
    } catch (error) {
        console.error('Connection test failed:', error);
        return false;
    }
}

// Inizializzazione
document.addEventListener('DOMContentLoaded', async () => {
    await loadSettings();

    document.getElementById('saveBtn').addEventListener('click', async () => {
        const config = await saveSettings();

        // Testa connessione
        const serverOk = await testConnection(config.serverUrl);

        if (serverOk) {
            showStatus('Impostazioni salvate! Server raggiungibile.');
        } else {
            showStatus('Salvato, ma server non raggiungibile. Verifica URL.', true);
        }
    });
});
